//Class to provide storage and other functionality for any warnings which the
//exporter generates.
#include "stdafx.h"
#include "warnings.h"
#include "osutils.h"
#include "rwcore.h"

CWarning::CWarning() {
}

CWarning::CWarning(TWarningType Type, CString NodeName, CString Message) {
    FWarningType=Type;
    FNodeName=NodeName;
    FMessage=Message;
};

CWarning::~CWarning( ) {
};

CString CWarning::getNodeName( void ) {
    return FNodeName;
}

CString CWarning::getMessage( void ) {
    return FMessage;
}

TWarningType CWarning::getType( void ) {
    return FWarningType;
}

WarningList::WarningList( ) {
    FNumInformationals=0;
    FNumWarnings=0;
    FNumErrors=0;
    zeroMetrics();
}

WarningList::~WarningList( ) {
}

void WarningList::add( CWarning &warning ) {
    FWarningList.AddTail(warning);
    //keep the count up to date every time a new warning is added
    switch (warning.getType()) {
        case wtInformational : FNumInformationals++; break;
        case wtWarning       : FNumWarnings++; break;
        case wtError         : FNumErrors++; break;
    }
}

void WarningList::saveToFile( FILE *fd ) {
    CWarning warning;
    CString node,msg;
    int i;

    try {
        for (i=0; i<FWarningList.GetCount(); i++) {
            warning=FWarningList.GetAt(FWarningList.FindIndex(i));
            node=warning.getNodeName();
            msg=warning.getMessage();
            switch (warning.getType()) {
                case wtInformational : fprintf(fd,"Informational,%s,%s\n",node,msg); break;
                case wtWarning       : fprintf(fd,"Warning,%s,%s\n",node,msg); break;
                case wtError         : fprintf(fd,"Error,%s,%s\n",node,msg); break;
            }
        }
    }
    catch(...) {
    }
}

int WarningList::getNumWarnings( void ) {
    return FWarningList.GetCount();
}

CWarning WarningList::getElement( int element ) {
    return FWarningList.GetAt(FWarningList.FindIndex(element));
}

int WarningList::numInformationals( void ) {
  return FNumInformationals;
}

int WarningList::numWarnings( void ) {
  return FNumWarnings;
}
int WarningList::numErrors( void ) {
  return FNumErrors;
}

void WarningList::zeroMetrics()
{
    m_ClumpCount=0;
    m_FrameCount=0;
    m_AtomicCount=0;
    m_NodeCount=0;
    m_TriangleCount=0;
    m_VertexCount=0;
    m_RotKeys=0;
    m_TransKeys=0;
}

void WarningList::writeVersion()
{  
    //Write out the version information for the file
    string version=GetFileVersion("rw30exp.dle", MAKELANGID(LANG_ENGLISH,SUBLANG_ENGLISH_UK));
    if (version.length()>0) {
        version="Exporter version "+version;
        add(CWarning(wtInformational,"",version.c_str()));
    }
    else {
        add(CWarning(wtInformational,"","Exporter version could not be determined"));
    }

    //determine which version of the renderware library this was built against
    RwInt32 rwversionint=RwEngineGetVersion();
    char VersionBuffer[255];
    sprintf(VersionBuffer, "Renderware V%x.%x%x", 
	    (RwEngineGetVersion() & 0xff00) >> 8,
	    (RwEngineGetVersion() & 0xf0) >> 4, 
	    (RwEngineGetVersion() & 0xf));
    add(CWarning(wtInformational,"",VersionBuffer));

    //and finally write out which version of max is being run
    sprintf(VersionBuffer,"3DS Max Version %i.%i.%i.%i",
        GET_MAX_RELEASE(Get3DSMAXVersion()),
        GET_MAX_API_NUM(Get3DSMAXVersion()),
        GET_MAX_SDK_REV(Get3DSMAXVersion()),
        GET_MAX_SDK_NUMREV(Get3DSMAXVersion()));
    add(CWarning(wtInformational,"",VersionBuffer));
}

void WarningList::newFrame( const char *NodeName )
{
    /* Log the creation of a new frame. */
    add(CWarning(wtInformational,NodeName,_T("Created a new frame")));
    /* Update global counts. */
    m_FrameCount+=1;
}

void WarningList::newClump( const char *NodeName )
{
    /* Log the creation of a new clump. */
    add(CWarning(wtInformational,NodeName,_T("Created a new clump")));
    /* Update global counts. */
    m_ClumpCount+=1;
}

void WarningList::writtenFile( const char *Filename )
{
    /* Log the creation of an export file - dff, bsp or ska */
    const int OneMegabyte=1024*1024;
    const int OneKilobyte=1024;
    WIN32_FIND_DATA finddata;
    HANDLE hFindFile;
    char temp[256];

    hFindFile=FindFirstFile( Filename, &finddata );
    if (hFindFile)
        FindClose(hFindFile);
    if (finddata.nFileSizeLow>OneMegabyte) {
        sprintf(temp,"Written file %s, size=%i MB (%i)",
            Filename,finddata.nFileSizeLow/OneMegabyte,finddata.nFileSizeLow);
    }
    else if (finddata.nFileSizeLow>OneKilobyte) {
        sprintf(temp,"Written file %s, size=%i KB (%i)",
            Filename,finddata.nFileSizeLow/OneKilobyte,finddata.nFileSizeLow);
    }
    else {
        sprintf(temp,"Written file %s, size=%i bytes",
            Filename,finddata.nFileSizeLow);
    }
    add(CWarning(wtInformational,"",temp));
}

